jQuery(document).ready(($) => {
  // Range slider value display
  $(".tiny-king-range").on("input", function () {
    $(this)
      .next(".tiny-king-range-value")
      .text($(this).val() + "%")
  })

  // Toggle resize dimensions visibility based on checkbox
  $('input[name="tiny_king_settings[resize_original]"]')
    .on("change", function () {
      if ($(this).is(":checked")) {
        $(".tiny-king-resize-dimensions").show()
      } else {
        $(".tiny-king-resize-dimensions").hide()
      }
    })
    .trigger("change")

  // Test connection button
  $("#tiny-king-test-connection").on("click", function () {
    var $button = $(this)
    var $result = $("#tiny-king-test-result")

    // Disable button and show loading
    $button.prop("disabled", true).text(tinyKingData.testingConnection)
    $result.html("").show()

    // Send AJAX request
    $.ajax({
      url: tinyKingData.ajaxUrl,
      type: "POST",
      data: {
        action: "tiny_king_test_connection",
        nonce: tinyKingData.nonce,
      },
      success: (response) => {
        if (response.success) {
          $result.html('<div class="notice notice-success inline"><p>' + response.data.message + "</p></div>")
        } else {
          $result.html(
            '<div class="notice notice-error inline"><p>' +
              tinyKingData.connectionFailed +
              response.data.message +
              "</p></div>",
          )
        }
      },
      error: () => {
        $result.html('<div class="notice notice-error inline"><p>' + tinyKingData.errorMessage + "</p></div>")
      },
      complete: () => {
        $button.prop("disabled", false).text(tinyKingData.testConnection)
      },
    })
  })

  // Check if images are already optimized and update buttons accordingly
  function checkOptimizedImages() {
    // Find all optimize buttons
    $(".tiny-king-optimize").each(function () {
      var $button = $(this)
      var id = $button.data("id")
      var $status = $("#tiny-king-status-" + id)

      // Check if there's optimization info for this image
      var $optimizationInfo = $button.closest("tr, .compat-field-tiny_king").find(".tiny-king-optimization-info")

      if ($optimizationInfo.length > 0) {
        // Image is already optimized, change button text
        $button.text(tinyKingData.reOptimize)
      }
    })
  }

  // Run the check when page loads
  checkOptimizedImages()

  // Optimize button in media library
  $(document).on("click", ".tiny-king-optimize", function () {
    var $button = $(this)
    var id = $button.data("id")
    var $status = $("#tiny-king-status-" + id)

    // Find optimization info section
    var $optimizationInfo = $button.closest("tr, .compat-field-tiny_king").find(".tiny-king-optimization-info")
    var $fileInfo = $button.closest("tr, .compat-field-tiny_king").find(".tiny-king-file-info")

    // Disable button and show loading
    $button.prop("disabled", true).text(tinyKingData.optimizing)
    $status
      .html(
        '<div class="tiny-king-loading"><span class="spinner is-active"></span> ' + tinyKingData.optimizing + "</div>",
      )
      .show()

    // Send AJAX request
    $.ajax({
      url: tinyKingData.ajaxUrl,
      type: "POST",
      data: {
        action: "tiny_king_optimize_image",
        nonce: tinyKingData.nonce,
        id: id,
      },
      success: (response) => {
        if (response.success) {
          // Show success message
          $status.html('<div class="tiny-king-success">' + response.data.message + "</div>")

          // Get optimization data
          var data = response.data.data
          if (data) {
            var savings = data.original_size - data.compressed_size
            var percent = data.savings_percent
            var timestamp = data.timestamp || Math.floor(Date.now() / 1000)
            var formattedDate = new Date(timestamp * 1000).toLocaleString()

            // Show optimization details in status area
            $status.append(
              '<div class="tiny-king-details">' +
                "<p>Original: " +
                formatBytes(data.original_size) +
                "</p>" +
                "<p>Compressed: " +
                formatBytes(data.compressed_size) +
                "</p>" +
                "<p>Savings: " +
                formatBytes(savings) +
                " (" +
                percent +
                "%)</p>" +
                "</div>",
            )

            // Update or create optimization info section
            var optimizationHtml =
              '<div class="tiny-king-optimization-info">' +
              "<p><strong>" +
              tinyKingData.originalSize +
              "</strong> " +
              formatBytes(data.original_size) +
              "</p>" +
              "<p><strong>" +
              tinyKingData.compressedSize +
              "</strong> " +
              formatBytes(data.compressed_size) +
              "</p>" +
              "<p><strong>" +
              tinyKingData.savings +
              "</strong> " +
              formatBytes(savings) +
              " (" +
              percent +
              "%)</p>" +
              "<p><strong>" +
              tinyKingData.optimizedOn +
              "</strong> " +
              formattedDate +
              "</p>" +
              "</div>"

            // If optimization info already exists, replace it, otherwise create it
            if ($optimizationInfo.length) {
              $optimizationInfo.replaceWith(optimizationHtml)
            } else {
              // Add after file info
              if ($fileInfo.length) {
                $fileInfo.after(
                  '<div class="tiny-king-field"><div class="tiny-king-label"><label>' +
                    tinyKingData.optimizationInfo +
                    '</label></div><div class="tiny-king-value">' +
                    optimizationHtml +
                    "</div></div>",
                )
              }
            }

            // Update file size in the UI if it changed
            if ($fileInfo.length) {
              $fileInfo
                .find('p:contains("Actual File Size")')
                .html("<strong>" + tinyKingData.actualFileSize + "</strong> " + formatBytes(data.compressed_size))
            }
          }

          // Change button text to "Re-optimize"
          $button.text(tinyKingData.reOptimize)
        } else {
          // Show error message
          $status.html('<div class="tiny-king-error">' + tinyKingData.optimizeFailed + response.data.message + "</div>")
        }
      },
      error: () => {
        $status.html('<div class="tiny-king-error">' + tinyKingData.errorMessage + "</div>")
      },
      complete: () => {
        $button.prop("disabled", false)
      },
    })
  })

  // Bulk optimize button
  $("#tiny-king-bulk-optimize").on("click", function () {
    var $button = $(this)
    var action = $button.data("action")

    if (action === "optimize") {
      startBulkOptimization($button)
    } else if (action === "cancel") {
      cancelBulkOptimization($button)
    }
  })

  // Initialize bulk optimization page
  if ($(".tiny-king-bulk-page").length) {
    initBulkOptimizer()
  }

  // Start bulk optimization
  function startBulkOptimization($button) {
    $button.prop("disabled", true)

    $.ajax({
      url: tinyKingData.ajaxUrl,
      type: "POST",
      data: {
        action: "tiny_king_start_bulk_optimization",
        nonce: tinyKingData.nonce,
      },
      success: (response) => {
        if (response.success) {
          $button.data("action", "cancel").text("CANCEL")
          $button.prop("disabled", false)

          // Clear the images list
          $("#tiny-king-images-list").empty()

          // Start processing images
          processNextImage()
        } else {
          $button.prop("disabled", false)
          alert(response.data.message || tinyKingData.errorMessage)
        }
      },
      error: () => {
        $button.prop("disabled", false)
        alert(tinyKingData.errorMessage)
      },
    })
  }

  // Cancel bulk optimization
  function cancelBulkOptimization($button) {
    $button.prop("disabled", true)

    $.ajax({
      url: tinyKingData.ajaxUrl,
      type: "POST",
      data: {
        action: "tiny_king_cancel_bulk_optimization",
        nonce: tinyKingData.nonce,
      },
      success: (response) => {
        if (response.success) {
          $button.data("action", "optimize").text("OPTIMIZE")
          $button.prop("disabled", false)

          // Update UI
          updateBulkOptimizationUI(false)
        } else {
          $button.prop("disabled", false)
          alert(response.data.message || tinyKingData.errorMessage)
        }
      },
      error: () => {
        $button.prop("disabled", false)
        alert(tinyKingData.errorMessage)
      },
    })
  }

  // Process next image
  function processNextImage() {
    $.ajax({
      url: tinyKingData.ajaxUrl,
      type: "POST",
      data: {
        action: "tiny_king_process_next_image",
        nonce: tinyKingData.nonce,
      },
      success: (response) => {
        if (response.success) {
          // Add image to the list
          if (response.data.result) {
            addImageToList(response.data.result)
          }

          // Update UI
          updateBulkOptimizationUI(true)

          // Check if complete
          if (response.data.is_complete) {
            // All done
            $("#tiny-king-bulk-optimize").data("action", "optimize").text("OPTIMIZE")
          } else {
            // Process next image
            setTimeout(processNextImage, 1000)
          }
        } else {
          alert(response.data.message || tinyKingData.errorMessage)
          $("#tiny-king-bulk-optimize").data("action", "optimize").text("OPTIMIZE")
        }
      },
      error: () => {
        alert(tinyKingData.errorMessage)
        $("#tiny-king-bulk-optimize").data("action", "optimize").text("OPTIMIZE")
      },
    })
  }

  // Add image to the list
  function addImageToList(image) {
    var statusHtml = ""

    if (image.status === "optimized") {
      statusHtml =
        '<div class="tiny-king-status-optimized"><span class="dashicons dashicons-yes"></span> Optimized</div>'
    } else if (image.status === "already_optimized") {
      statusHtml =
        '<div class="tiny-king-status-already-optimized"><span class="dashicons dashicons-yes"></span> Already optimized</div>'
    } else if (image.status === "error") {
      statusHtml = '<div class="tiny-king-status-error"><span class="dashicons dashicons-no"></span> Error</div>'
    } else {
      statusHtml = '<div class="tiny-king-status-compressing"><div class="tiny-king-spinner"></div> Compressing</div>'
    }

    var html =
      '<tr id="tiny-king-image-' +
      image.id +
      '">' +
      "<td>" +
      '<div class="tiny-king-image-cell">' +
      '<img src="' +
      image.thumbnail +
      '" alt="" class="tiny-king-image-thumbnail">' +
      '<span class="tiny-king-image-name">' +
      image.title +
      "</span>" +
      "</div>" +
      "</td>" +
      "<td>" +
      formatBytes(image.original_size) +
      "</td>" +
      "<td>" +
      formatBytes(image.compressed_size) +
      "</td>" +
      "<td>" +
      (image.savings_percent ? image.savings_percent.toFixed(1) : "0.0") +
      "%</td>" +
      "<td>" +
      statusHtml +
      "</td>" +
      "</tr>"

    // Add to the top of the list
    $("#tiny-king-images-list").prepend(html)
  }

  // Initialize bulk optimization page
  function initBulkOptimizationUI() {
    $.ajax({
      url: tinyKingData.ajaxUrl,
      type: "POST",
      data: {
        action: "tiny_king_get_bulk_status",
        nonce: tinyKingData.nonce,
      },
      success: (response) => {
        if (response.success) {
          // Update UI based on status
          updateBulkOptimizationUI(response.data.is_running)

          // Update button
          if (response.data.is_running) {
            $("#tiny-king-bulk-optimize").data("action", "cancel").text("CANCEL")
          } else {
            $("#tiny-king-bulk-optimize").data("action", "optimize").text("OPTIMIZE")
          }

          // Add processed images to the list
          if (response.data.processed_images && response.data.processed_images.length > 0) {
            $("#tiny-king-images-list").empty()

            // Get image details and add to list
            response.data.processed_images.forEach((processedImage) => {
              // In a real implementation, you would fetch image details here
              // For now, we'll just add a placeholder
              var image = {
                id: processedImage.id,
                title: "Image #" + processedImage.id,
                thumbnail: "", // Would need to fetch this
                original_size: processedImage.original_size || 0,
                compressed_size: processedImage.compressed_size || 0,
                savings_percent: processedImage.savings_percent || 0,
                status: processedImage.status,
              }

              addImageToList(image)
            })
          }

          // If optimization is running, continue processing
          if (response.data.is_running) {
            processNextImage()
          }
        }
      },
    })
  }

  // Update bulk optimization UI
  function updateBulkOptimizationUI(isRunning) {
    // This would update progress bars, stats, etc.
    // For a complete implementation, you would fetch current stats
    // and update all UI elements accordingly

    // For now, we'll just update the button state
    var $button = $("#tiny-king-bulk-optimize")

    if (isRunning) {
      $button.data("action", "cancel").text("CANCEL")
    } else {
      $button.data("action", "optimize").text("OPTIMIZE")
    }
  }

  // Initialize bulk optimization page
  function initBulkOptimizationPage() {
    // Initialize the UI
    initBulkOptimizationUI()

    // Initialize the chart
    initSavingsChart()
  }

  // Initialize savings chart
  function initSavingsChart() {
    var $chart = $(".tiny-king-chart-circle")

    if ($chart.length) {
      var percent = Number.parseFloat($chart.data("percent")) || 0

      // Ensure the percentage is properly rounded to 1 decimal place
      percent = Math.round(percent * 10) / 10

      // Update the percentage display in the chart
      $chart.find(".tiny-king-chart-percent").text(percent.toFixed(1) + "%")

      // For a real implementation, you would use a proper chart library
      // For now, we'll just set a CSS variable for a simple circle chart
      $chart.css("background", "conic-gradient(#2271b1 " + percent + "%, #f0f0f1 0%)")
    }
  }

  // Format bytes to human-readable size
  function formatBytesOriginal(bytes, decimals = 1) {
    if (bytes === 0) return "0 Bytes"

    const k = 1024
    const dm = decimals < 0 ? 0 : decimals
    const sizes = ["Bytes", "KB", "MB", "GB", "TB"]

    const i = Math.floor(Math.log(bytes) / Math.log(k))

    return Number.parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + " " + sizes[i]
  }

  // Format bytes to human-readable format
  function formatBytes(bytes, decimals = 2) {
    if (bytes === 0) return "0 Bytes"

    const k = 1024
    const dm = decimals < 0 ? 0 : decimals
    const sizes = ["Bytes", "KB", "MB", "GB", "TB", "PB", "EB", "ZB", "YB"]

    const i = Math.floor(Math.log(bytes) / Math.log(k))

    return Number.parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + " " + sizes[i]
  }

  // Initialize bulk optimizer
  function initBulkOptimizer() {
    var $button = $("#tiny-king-bulk-optimize")
    var $progressBar = $(".tiny-king-progress-bar")
    var $progressText = $(".tiny-king-progress-text")
    var $imagesList = $("#tiny-king-images-list")
    var isRunning = false

    // Start or cancel bulk optimization
    $button.on("click", () => {
      var action = $button.data("action")

      if (action === "optimize") {
        startBulkOptimization()
      } else if (action === "cancel") {
        cancelBulkOptimization()
      }
    })

    // Start bulk optimization
    function startBulkOptimization() {
      $button.prop("disabled", true)

      // Send AJAX request
      $.ajax({
        url: tinyKingData.ajaxUrl,
        type: "POST",
        data: {
          action: "tiny_king_start_bulk_optimization",
          nonce: tinyKingData.nonce,
        },
        success: (response) => {
          if (response.success) {
            isRunning = true
            $button.data("action", "cancel").text(tinyKingData.cancel)
            $imagesList.empty()

            // Process first image
            processNextImage()
          } else {
            alert(response.data.message)
          }
        },
        error: () => {
          alert(tinyKingData.errorMessage)
        },
        complete: () => {
          $button.prop("disabled", false)
        },
      })
    }

    // Cancel bulk optimization
    function cancelBulkOptimization() {
      $button.prop("disabled", true)

      // Send AJAX request
      $.ajax({
        url: tinyKingData.ajaxUrl,
        type: "POST",
        data: {
          action: "tiny_king_cancel_bulk_optimization",
          nonce: tinyKingData.nonce,
        },
        success: (response) => {
          if (response.success) {
            isRunning = false
            $button.data("action", "optimize").text(tinyKingData.optimize)
          } else {
            alert(response.data.message)
          }
        },
        error: () => {
          alert(tinyKingData.errorMessage)
        },
        complete: () => {
          $button.prop("disabled", false)
        },
      })
    }

    // Process next image
    function processNextImage() {
      if (!isRunning) {
        return
      }

      // Send AJAX request
      $.ajax({
        url: tinyKingData.ajaxUrl,
        type: "POST",
        data: {
          action: "tiny_king_process_next_image",
          nonce: tinyKingData.nonce,
        },
        success: (response) => {
          if (response.success) {
            // Update progress
            var status = response.data.status
            var progress = Math.round((status.processed_images / status.total_images) * 100)

            $progressBar.css("width", progress + "%")
            $progressText.text(status.processed_images + " / " + status.total_images + " (" + progress + "%)")

            // Add image to list
            if (response.data.result) {
              addImageToList(response.data.result)
            }

            // Check if complete
            if (response.data.complete) {
              isRunning = false
              $button.data("action", "optimize").text(tinyKingData.optimize)
              alert(tinyKingData.bulkComplete)
            } else {
              // Process next image
              setTimeout(processNextImage, 500)
            }
          } else {
            isRunning = false
            $button.data("action", "optimize").text(tinyKingData.optimize)
            alert(tinyKingData.bulkFailed + ": " + response.data.message)
          }
        },
        error: () => {
          isRunning = false
          $button.data("action", "optimize").text(tinyKingData.optimize)
          alert(tinyKingData.errorMessage)
        },
      })
    }

    // Add image to list
    function addImageToList(result) {
      var savings = result.original_size - result.compressed_size
      var savingsPercent = result.savings_percent
      var statusClass = ""
      var statusText = ""

      // Determine status text and class
      if (result.status === "optimized") {
        statusClass = "tiny-king-status-success"
        statusText = "✓ " + tinyKingData.optimizeSuccess
      } else if (result.status === "already_optimized") {
        statusClass = "tiny-king-status-already-optimized"
        statusText = "✓ Already optimized"
      } else if (result.status === "error") {
        statusClass = "tiny-king-status-error"
        statusText = "✗ " + result.message
      }

      // Create row
      var $row = $("<tr></tr>")

      // Add cells
      $row.append('<td><img src="' + result.thumbnail + '" width="40" height="40" alt="" /> ' + result.title + "</td>")
      $row.append("<td>" + formatBytes(result.original_size) + "</td>")
      $row.append("<td>" + formatBytes(result.compressed_size) + "</td>")
      $row.append("<td>" + (typeof savingsPercent === "number" ? savingsPercent.toFixed(1) : savingsPercent) + "%</td>")
      $row.append('<td><span class="' + statusClass + '">' + statusText + "</span></td>")

      // Add to table
      $imagesList.prepend($row)
    }

    // Update bulk optimizer UI
    function updateBulkOptimizerUI() {
      // Send AJAX request
      $.ajax({
        url: tinyKingData.ajaxUrl,
        type: "POST",
        data: {
          action: "tiny_king_get_bulk_status",
          nonce: tinyKingData.nonce,
        },
        success: (response) => {
          if (response.success) {
            var status = response.data.status

            // Update button
            if (status.is_running) {
              isRunning = true
              $button.data("action", "cancel").text(tinyKingData.cancel)
            } else {
              isRunning = false
              $button.data("action", "optimize").text(tinyKingData.optimize)
            }

            // Update progress
            var progress = Math.round((status.processed_images / status.total_images) * 100)
            $progressBar.css("width", progress + "%")
            $progressText.text(status.processed_images + " / " + status.total_images + " (" + progress + "%)")

            // Update image list
            $imagesList.empty()

            if (status.results) {
              // Convert object to array and sort by ID
              var results = []
              for (var id in status.results) {
                results.push(status.results[id])
              }

              results.sort((a, b) => b.id - a.id)

              // Add each result to the list
              for (var i = 0; i < results.length; i++) {
                addImageToList(results[i])
              }
            }

            // Continue processing if running
            if (isRunning) {
              processNextImage()
            }
          }
        },
      })
    }

    // Check if optimization is already running
    updateBulkOptimizerUI()
  }

  // API key validation
  $("#tiny-king-api-key").on("blur", function () {
    var $input = $(this)
    var apiKey = $input.val().trim()
    var $wrapper = $input.closest(".tiny-king-api-key-wrapper")
    var $indicator = $wrapper.find(".tiny-king-validation-indicator")
    var apiUrl = $('input[name="tiny_king_settings[api_url]"]').val().trim()

    // Remove any existing indicator
    if ($indicator.length) {
      $indicator.remove()
    }

    // Skip validation if empty
    if (!apiKey) {
      return
    }

    // Check if API URL is provided
    if (!apiUrl) {
      $wrapper.append(
        '<span class="tiny-king-validation-indicator dashicons dashicons-warning unknown" title="' +
          tinyKingData.apiUrlRequired +
          '"></span>',
      )
      return
    }

    // Add a loading indicator
    $wrapper.append('<span class="tiny-king-validation-indicator dashicons dashicons-update-alt spinning"></span>')

    // Send AJAX request to validate
    $.ajax({
      url: tinyKingData.ajaxUrl,
      type: "POST",
      data: {
        action: "tiny_king_validate_api_key",
        nonce: tinyKingData.nonce,
        api_key: apiKey,
        api_url: apiUrl,
      },
      success: (response) => {
        if (response.success) {
          $wrapper
            .find(".tiny-king-validation-indicator")
            .removeClass("dashicons-update-alt spinning")
            .addClass("dashicons-yes valid")
            .attr("title", tinyKingData.apiKeyValid)
        } else {
          $wrapper
            .find(".tiny-king-validation-indicator")
            .removeClass("dashicons-update-alt spinning")
            .addClass("dashicons-no invalid")
            .attr("title", tinyKingData.apiKeyInvalid + ": " + response.data.message)
        }
      },
      error: () => {
        $wrapper
          .find(".tiny-king-validation-indicator")
          .removeClass("dashicons-update-alt spinning")
          .addClass("dashicons-no invalid")
          .attr("title", tinyKingData.connectionError)
      },
    })
  })

  // Add a spinning animation for the update icon
  $("<style>")
    .prop("type", "text/css")
    .html(`
      .dashicons.spinning {
        animation: tiny-king-spin 2s linear infinite;
      }
      @keyframes tiny-king-spin {
        0% { transform: translateY(-50%) rotate(0deg); }
        100% { transform: translateY(-50%) rotate(360deg); }
      }
    `)
    .appendTo("head")
})
