<?php
/**
 * AJAX handlers for TinyKing
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Test API connection
 */
function tiny_king_test_connection() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'tiny-king-nonce')) {
        wp_send_json_error(['message' => __('Security check failed.', 'tiny-king')]);
    }
    
    // Get settings
    $settings = get_option('tiny_king_settings', []);
    $api_url = isset($settings['api_url']) ? $settings['api_url'] : '';
    $api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
    
    // Check if API settings are configured
    if (empty($api_url) || empty($api_key)) {
        // Store validation status as invalid
        set_transient('tiny_king_api_key_validation', 'invalid', DAY_IN_SECONDS);
        wp_send_json_error(['message' => __('API URL and API key must be configured.', 'tiny-king')]);
    }
    
    // Make a test request
    $response = wp_remote_post($api_url, [
        'headers' => [
            'x-api-key' => $api_key,
        ],
        'body' => [
            'test' => 'connection',
        ],
        'timeout' => 30,
        'sslverify' => apply_filters('tiny_king_sslverify', true),
    ]);
    
    // Check for errors
    if (is_wp_error($response)) {
        // Store validation status as invalid
        set_transient('tiny_king_api_key_validation', 'invalid', DAY_IN_SECONDS);
        wp_send_json_error(['message' => $response->get_error_message()]);
    }
    
    // Check response code
    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        // Store validation status as invalid
        set_transient('tiny_king_api_key_validation', 'invalid', DAY_IN_SECONDS);
        wp_send_json_error(['message' => sprintf(__('HTTP error: %d', 'tiny-king'), $response_code)]);
    }
    
    // Parse response body
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);
    
    // Check if response indicates success
    if (!$data || !isset($data['success']) || !$data['success']) {
        // Store validation status as invalid
        set_transient('tiny_king_api_key_validation', 'invalid', DAY_IN_SECONDS);
        wp_send_json_error(['message' => __('Invalid API response.', 'tiny-king')]);
    }
    
    // Store validation status as valid
    set_transient('tiny_king_api_key_validation', 'valid', DAY_IN_SECONDS);
    
    // Success
    wp_send_json_success(['message' => __('Connection successful!', 'tiny-king')]);
}
add_action('wp_ajax_tiny_king_test_connection', 'tiny_king_test_connection');

/**
 * Validate API key without saving settings
 */
function tiny_king_validate_api_key() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'tiny-king-nonce')) {
        wp_send_json_error(['message' => __('Security check failed.', 'tiny-king')]);
    }
    
    // Get API key and URL from POST data
    $api_key = isset($_POST['api_key']) ? sanitize_text_field($_POST['api_key']) : '';
    $api_url = isset($_POST['api_url']) ? esc_url_raw($_POST['api_url']) : '';
    
    // Check if API settings are provided
    if (empty($api_url) || empty($api_key)) {
        wp_send_json_error(['message' => __('API URL and API key must be provided.', 'tiny-king')]);
    }
    
    // Make a test request
    $response = wp_remote_post($api_url, [
        'headers' => [
            'x-api-key' => $api_key,
        ],
        'body' => [
            'test' => 'connection',
        ],
        'timeout' => 15,
        'sslverify' => apply_filters('tiny_king_sslverify', true),
    ]);
    
    // Check for errors
    if (is_wp_error($response)) {
        wp_send_json_error(['message' => $response->get_error_message()]);
    }
    
    // Check response code
    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        wp_send_json_error(['message' => sprintf(__('HTTP error: %d', 'tiny-king'), $response_code)]);
    }
    
    // Parse response body
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);
    
    // Check if response indicates success
    if (!$data || !isset($data['success']) || !$data['success']) {
        wp_send_json_error(['message' => __('Invalid API response.', 'tiny-king')]);
    }
    
    // Success
    wp_send_json_success(['message' => __('API key is valid.', 'tiny-king')]);
}
add_action('wp_ajax_tiny_king_validate_api_key', 'tiny_king_validate_api_key');

/**
 * Optimize a single image
 */
function tiny_king_optimize_image() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'tiny-king-nonce')) {
        wp_send_json_error(['message' => __('Security check failed.', 'tiny-king')]);
    }
    
    // Check attachment ID
    if (!isset($_POST['id']) || !is_numeric($_POST['id'])) {
        wp_send_json_error(['message' => __('Invalid attachment ID.', 'tiny-king')]);
    }
    
    $attachment_id = intval($_POST['id']);
    
    // Check if attachment exists
    if (!get_post($attachment_id)) {
        wp_send_json_error(['message' => __('Attachment not found.', 'tiny-king')]);
    }
    
    // Get TinyKing instance
    global $tiny_king;
    if (!$tiny_king) {
        wp_send_json_error(['message' => __('TinyKing plugin not initialized.', 'tiny-king')]);
    }
    
    // Optimize the image
    $result = $tiny_king->compress_attachment($attachment_id);
    
    // Check for errors
    if (is_wp_error($result)) {
        wp_send_json_error(['message' => $result->get_error_message()]);
    }
    
    // Success - return complete optimization data
    wp_send_json_success([
        'message' => __('Image optimized successfully!', 'tiny-king'),
        'data' => $result,
    ]);
}
add_action('wp_ajax_tiny_king_optimize_image', 'tiny_king_optimize_image');

/**
 * Start bulk optimization
 */
function tiny_king_start_bulk_optimization() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'tiny-king-nonce')) {
        wp_send_json_error(['message' => __('Security check failed.', 'tiny-king')]);
    }
    
    // Get all image attachments
    $args = [
        'post_type' => 'attachment',
        'post_mime_type' => ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
        'post_status' => 'inherit',
        'posts_per_page' => -1,
        'fields' => 'ids',
    ];
    
    $image_ids = get_posts($args);
    
    // Set up optimization status
    $status = [
        'is_running' => true,
        'total_images' => count($image_ids),
        'processed_images' => 0,
        'queue' => $image_ids,
        'current_image' => null,
        'results' => [],
    ];
    
    // Store status in a transient
    set_transient('tiny_king_bulk_optimization_status', $status, 12 * HOUR_IN_SECONDS);
    
    // Return success
    wp_send_json_success([
        'message' => __('Bulk optimization started.', 'tiny-king'),
        'status' => $status,
    ]);
}
add_action('wp_ajax_tiny_king_start_bulk_optimization', 'tiny_king_start_bulk_optimization');

/**
 * Cancel bulk optimization
 */
function tiny_king_cancel_bulk_optimization() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'tiny-king-nonce')) {
        wp_send_json_error(['message' => __('Security check failed.', 'tiny-king')]);
    }
    
    // Get current status
    $status = get_transient('tiny_king_bulk_optimization_status');
    
    if ($status) {
        // Update status
        $status['is_running'] = false;
        
        // Store updated status
        set_transient('tiny_king_bulk_optimization_status', $status, 12 * HOUR_IN_SECONDS);
    }
    
    // Return success
    wp_send_json_success([
        'message' => __('Bulk optimization cancelled.', 'tiny-king'),
    ]);
}
add_action('wp_ajax_tiny_king_cancel_bulk_optimization', 'tiny_king_cancel_bulk_optimization');

/**
 * Process next image in bulk optimization
 */
function tiny_king_process_next_image() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'tiny-king-nonce')) {
        wp_send_json_error(['message' => __('Security check failed.', 'tiny-king')]);
    }
    
    // Get current status
    $status = get_transient('tiny_king_bulk_optimization_status');
    
    if (!$status || !$status['is_running']) {
        wp_send_json_error(['message' => __('Bulk optimization is not running.', 'tiny-king')]);
    }
    
    // Check if there are images in the queue
    if (empty($status['queue'])) {
        // All images processed
        $status['is_running'] = false;
        set_transient('tiny_king_bulk_optimization_status', $status, 12 * HOUR_IN_SECONDS);
        
        wp_send_json_success([
            'message' => __('All images processed.', 'tiny-king'),
            'status' => $status,
            'is_complete' => true,
        ]);
    }
    
    // Get next image from queue
    $attachment_id = array_shift($status['queue']);
    $status['current_image'] = $attachment_id;
    
    // Get TinyKing instance
    global $tiny_king;
    if (!$tiny_king) {
        wp_send_json_error(['message' => __('TinyKing plugin not initialized.', 'tiny-king')]);
    }
    
    // Get attachment metadata
    $attachment = get_post($attachment_id);
    $file_path = get_attached_file($attachment_id, true);
    $metadata = wp_get_attachment_metadata($attachment_id);
    
    // Check if image is already optimized
    $optimization_data = get_post_meta($attachment_id, '_tiny_king_data', true);
    $is_already_optimized = $optimization_data && isset($optimization_data['optimized']) && $optimization_data['optimized'];
    
    // Prepare result data
    $result = [
        'id' => $attachment_id,
        'title' => $attachment->post_title,
        'filename' => basename($file_path),
        'thumbnail' => wp_get_attachment_image_url($attachment_id, 'thumbnail'),
    ];
    
    if ($is_already_optimized) {
        // Use the stored optimization data for already optimized images
        $result['status'] = 'already_optimized';
        $result['original_size'] = $optimization_data['original_size'];
        $result['compressed_size'] = $optimization_data['compressed_size'];
        $result['savings'] = $optimization_data['savings'];
        $result['savings_percent'] = $optimization_data['savings_percent'];
    } else {
        // Optimize the image
        $optimization_result = $tiny_king->compress_attachment($attachment_id);
        
        if (is_wp_error($optimization_result)) {
            $result['status'] = 'error';
            $result['message'] = $optimization_result->get_error_message();
            $result['original_size'] = filesize($file_path);
            $result['compressed_size'] = filesize($file_path);
            $result['savings'] = 0;
            $result['savings_percent'] = 0;
        } else {
            $result['status'] = 'optimized';
            $result['original_size'] = $optimization_result['original_size'];
            $result['compressed_size'] = $optimization_result['compressed_size'];
            $result['savings'] = $optimization_result['savings'];
            $result['savings_percent'] = $optimization_result['savings_percent'];
        }
    }
    
    // Ensure consistent rounding to 1 decimal place
    if (isset($result['savings_percent']) && is_numeric($result['savings_percent'])) {
        $result['savings_percent'] = round($result['savings_percent'], 1);
    }
    
    // Update status
    $status['processed_images']++;
    $status['results'][$attachment_id] = $result;
    set_transient('tiny_king_bulk_optimization_status', $status, 12 * HOUR_IN_SECONDS);
    
    // Return result
    wp_send_json_success([
        'message' => __('Image processed.', 'tiny-king'),
        'status' => $status,
        'result' => $result,
        'is_complete' => empty($status['queue']),
    ]);
}
add_action('wp_ajax_tiny_king_process_next_image', 'tiny_king_process_next_image');

/**
 * Get bulk optimization status
 */
function tiny_king_get_bulk_status() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'tiny-king-nonce')) {
        wp_send_json_error(['message' => __('Security check failed.', 'tiny-king')]);
    }
    
    // Get current status
    $status = get_transient('tiny_king_bulk_optimization_status');
    
    if (!$status) {
        $status = [
            'is_running' => false,
            'total_images' => 0,
            'processed_images' => 0,
            'queue' => [],
            'current_image' => null,
            'results' => [],
        ];
    }
    
    // Return status
    wp_send_json_success([
        'status' => $status,
        'is_running' => $status['is_running'] ?? false,
    ]);
}
add_action('wp_ajax_tiny_king_get_bulk_status', 'tiny_king_get_bulk_status');
