<?php
/**
 * Plugin Name: TinyKing Image Optimizer
 * Plugin URI: https://tinyking.rkingbon.com
 * Description: Optimize your WordPress images using the TinyKing API
 * Version: 1.0.0
 * Author: You Tell Me -- lols
 * Author URI: https://rkingbon.com
 * Text Domain: tiny-king
 * License: GPL v2 or later
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('TINY_KING_VERSION', '1.0.0');
define('TINY_KING_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('TINY_KING_PLUGIN_URL', plugin_dir_url(__FILE__));

// Include AJAX handlers
require_once TINY_KING_PLUGIN_DIR . 'includes/ajax-handlers.php';

class TinyKing {
    /**
     * API endpoint URL
     * @var string
     */
    private $api_url;

    /**
     * API key for authentication
     * @var string
     */
    private $api_key;

    /**
     * Plugin settings
     * @var array
     */
    private $settings;

    /**
     * Constructor
     */
    public function __construct() {
        // Initialize settings
        $this->settings = get_option('tiny_king_settings', [
            'api_url' => '',
            'api_key' => '',
            'quality' => 65, // Lower default quality for better compression
            'format' => 'original',
            'compression_mode' => 'background', // background, upload, manual
            'image_sizes' => [
                'original' => true,
                'thumbnail' => true,
                'medium' => true,
                'medium_large' => true,
                'large' => true,
                '1536x1536' => true,
                '2048x2048' => true,
            ],
            'resize_original' => false,
            'max_width' => 2048,
            'max_height' => 2048,
            'preserve_exif_data' => false,
            'preserve_copyright' => false,
            'preserve_gps' => false,
        ]);

        $this->api_url = $this->settings['api_url'];
        $this->api_key = $this->settings['api_key'];

        // Register hooks
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        
        // Add media library integration
        add_filter('attachment_fields_to_edit', [$this, 'add_compress_button'], 10, 2);
        
        // Add filter to display actual file size in media library
        add_filter('media_row_actions', [$this, 'add_file_size_to_media_row'], 10, 2);
        
        // Auto-optimize on upload if enabled
        if ($this->settings['compression_mode'] === 'upload') {
            add_filter('wp_handle_upload', [$this, 'auto_compress_uploaded_image']);
        }
        
        // Background optimization
        if ($this->settings['compression_mode'] === 'background') {
            add_action('add_attachment', [$this, 'schedule_background_optimization']);
            add_action('tiny_king_background_optimize', [$this, 'process_background_optimization']);
        }
        
        // Add debug info to attachment details
        add_filter('attachment_fields_to_edit', [$this, 'add_file_info_to_attachment'], 10, 2);
    }

    /**
     * Schedule background optimization for a new attachment
     */
    public function schedule_background_optimization($attachment_id) {
        if (wp_attachment_is_image($attachment_id)) {
            wp_schedule_single_event(time() + 10, 'tiny_king_background_optimize', [$attachment_id]);
        }
    }

    /**
     * Process background optimization
     */
    public function process_background_optimization($attachment_id) {
        if (wp_attachment_is_image($attachment_id)) {
            $this->compress_attachment($attachment_id);
            
            // Process all selected image sizes
            $this->compress_image_sizes($attachment_id);
        }
    }

    /**
     * Compress all selected image sizes for an attachment
     */
    public function compress_image_sizes($attachment_id) {
        $metadata = wp_get_attachment_metadata($attachment_id);
        
        if (!isset($metadata['sizes']) || !is_array($metadata['sizes'])) {
            return;
        }
        
        $upload_dir = wp_upload_dir();
        $base_dir = $upload_dir['basedir'] . '/' . dirname($metadata['file']) . '/';
        
        foreach ($metadata['sizes'] as $size_name => $size_data) {
            // Check if this size should be compressed
            if (!isset($this->settings['image_sizes'][$size_name]) || !$this->settings['image_sizes'][$size_name]) {
                continue;
            }
            
            $file_path = $base_dir . $size_data['file'];
            
            if (file_exists($file_path)) {
                // Compress this size
                $this->compress_single_file($file_path);
            }
        }
    }

    /**
     * Compress a single file without attachment context
     */
    public function compress_single_file($file_path) {
        if (!file_exists($file_path)) {
            return false;
        }
        
        // Check API settings
        if (empty($this->api_url) || empty($this->api_key)) {
            return false;
        }
        
        // Get file info
        $file_name = basename($file_path);
        $original_size = filesize($file_path);
        $mime_type = mime_content_type($file_path);
        
        // Use cURL for direct file upload
        $curl = curl_init();
        
        // Create a CURLFile object
        $cfile = curl_file_create($file_path, $mime_type, $file_name);
        
        // Set up the POST fields
        $post_fields = [
            'image' => $cfile,
            'format' => isset($this->settings['format']) ? $this->settings['format'] : 'original',
            'quality' => isset($this->settings['quality']) ? $this->settings['quality'] : 65,
            'stripMetadata' => !($this->settings['preserve_exif_data'] || $this->settings['preserve_copyright'] || $this->settings['preserve_gps']),
            'forceCompression' => 'true'
        ];
        
        // Set cURL options
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->api_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $post_fields,
            CURLOPT_HTTPHEADER => [
                'x-api-key: ' . $this->api_key,
            ],
            CURLOPT_TIMEOUT => 60,
            CURLOPT_SSL_VERIFYPEER => apply_filters('tiny_king_sslverify', true),
        ]);
        
        // Execute the request
        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        
        curl_close($curl);
        
        // Check for errors
        if ($response === false || $http_code !== 200) {
            return false;
        }
        
        // Parse the JSON response
        $data = json_decode($response, true);
        
        if (!$data || !isset($data['success']) || !$data['success']) {
            return false;
        }
        
        // Get the compressed image data from base64
        if (!isset($data['base64Data']) || empty($data['base64Data'])) {
            return false;
        }
        
        $base64_data = $data['base64Data'];
        // Remove data URL prefix if present
        if (strpos($base64_data, 'data:') === 0) {
            $base64_data = substr($base64_data, strpos($base64_data, ',') + 1);
        }
        $compressed_data = base64_decode($base64_data);
        
        // Create a backup of the original file
        $backup_path = $file_path . '.bak';
        if (!file_exists($backup_path)) {
            copy($file_path, $backup_path);
        }
        
        // Save the compressed image
        $saved = file_put_contents($file_path, $compressed_data);
        if (!$saved) {
            return false;
        }
        
        // Verify the file was actually written and get its size
        clearstatcache(true, $file_path);
        $compressed_size = filesize($file_path);
        
        // If the compression didn't reduce the file size, restore the original
        if ($compressed_size >= $original_size) {
            // Restore the original file
            if (file_exists($backup_path)) {
                copy($backup_path, $file_path);
            }
            return false;
        }
        
        return true;
    }

    /**
     * Add admin menu pages
     */
    public function add_admin_menu() {
        add_menu_page(
            __('TinyKing', 'tiny-king'),
            __('TinyKing', 'tiny-king'),
            'manage_options',
            'tiny-king',
            [$this, 'render_settings_page'],
            'dashicons-image-filter',
            81
        );
        
        add_submenu_page(
            'tiny-king',
            __('Settings', 'tiny-king'),
            __('Settings', 'tiny-king'),
            'manage_options',
            'tiny-king',
            [$this, 'render_settings_page']
        );
        
        add_submenu_page(
            'tiny-king',
            __('Bulk Optimizer', 'tiny-king'),
            __('Bulk Optimizer', 'tiny-king'),
            'manage_options',
            'tiny-king-bulk',
            [$this, 'render_bulk_optimizer_page']
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting(
            'tiny_king_settings', 
            'tiny_king_settings',
            [
                'sanitize_callback' => [$this, 'validate_api_key'],
            ]
        );
        
        // API Settings Section
        add_settings_section(
            'tiny_king_api_settings',
            __('API Settings', 'tiny-king'),
            [$this, 'render_api_settings_section'],
            'tiny_king_settings'
        );
        
        add_settings_field(
            'api_url',
            __('API URL', 'tiny-king'),
            [$this, 'render_api_url_field'],
            'tiny_king_settings',
            'tiny_king_api_settings'
        );
        
        add_settings_field(
            'api_key',
            __('API Key', 'tiny-king'),
            [$this, 'render_api_key_field'],
            'tiny_king_settings',
            'tiny_king_api_settings'
        );
        
        // New Image Uploads Section
        add_settings_section(
            'tiny_king_upload_settings',
            __('New Image Uploads', 'tiny-king'),
            [$this, 'render_upload_settings_section'],
            'tiny_king_settings'
        );
        
        add_settings_field(
            'compression_mode',
            __('When should new images be compressed?', 'tiny-king'),
            [$this, 'render_compression_mode_field'],
            'tiny_king_settings',
            'tiny_king_upload_settings'
        );
        
        // Image Sizes Section
        add_settings_section(
            'tiny_king_sizes_settings',
            __('Image Sizes', 'tiny-king'),
            [$this, 'render_sizes_settings_section'],
            'tiny_king_settings'
        );
        
        add_settings_field(
            'image_sizes',
            __('Select image sizes to be compressed', 'tiny-king'),
            [$this, 'render_image_sizes_field'],
            'tiny_king_settings',
            'tiny_king_sizes_settings'
        );
        
        // Original Image Section
        add_settings_section(
            'tiny_king_original_settings',
            __('Original Image', 'tiny-king'),
            [$this, 'render_original_settings_section'],
            'tiny_king_settings'
        );
        
        add_settings_field(
            'resize_original',
            __('Resize the original image', 'tiny-king'),
            [$this, 'render_resize_original_field'],
            'tiny_king_settings',
            'tiny_king_original_settings'
        );
        
        add_settings_field(
            'preserve_metadata',
            __('Preserve metadata', 'tiny-king'),
            [$this, 'render_preserve_metadata_field'],
            'tiny_king_settings',
            'tiny_king_original_settings'
        );
        
        // Optimization Settings Section
        add_settings_section(
            'tiny_king_optimization_settings',
            __('Optimization Settings', 'tiny-king'),
            [$this, 'render_optimization_settings_section'],
            'tiny_king_settings'
        );
        
        add_settings_field(
            'quality',
            __('Quality', 'tiny-king'),
            [$this, 'render_quality_field'],
            'tiny_king_settings',
            'tiny_king_optimization_settings'
        );
        
        add_settings_field(
            'format',
            __('Format', 'tiny-king'),
            [$this, 'render_format_field'],
            'tiny_king_settings',
            'tiny_king_optimization_settings'
        );
    }

    /**
     * Validate API key when settings are saved
     */
    public function validate_api_key($settings) {
        // Check if API key has changed
        $old_settings = get_option('tiny_king_settings', []);
        $old_api_key = isset($old_settings['api_key']) ? $old_settings['api_key'] : '';
        $new_api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
        
        // If API key has changed, reset validation status
        if ($old_api_key !== $new_api_key) {
            delete_transient('tiny_king_api_key_validation');
            
            // If new API key is not empty, try to validate it
            if (!empty($new_api_key) && !empty($settings['api_url'])) {
                $this->validate_api_key_request($settings['api_url'], $new_api_key);
            }
        }
        
        return $settings;
    }

    /**
     * Make a request to validate API key
     */
    private function validate_api_key_request($api_url, $api_key) {
        // Make a test request
        $response = wp_remote_post($api_url, [
            'headers' => [
                'x-api-key' => $api_key,
            ],
            'body' => [
                'test' => 'connection',
            ],
            'timeout' => 15,
            'sslverify' => apply_filters('tiny_king_sslverify', true),
        ]);
        
        // Check for errors
        if (is_wp_error($response)) {
            set_transient('tiny_king_api_key_validation', 'invalid', DAY_IN_SECONDS);
            return false;
        }
        
        // Check response code
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            set_transient('tiny_king_api_key_validation', 'invalid', DAY_IN_SECONDS);
            return false;
        }
        
        // Parse response body
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        // Check if response indicates success
        if (!$data || !isset($data['success']) || !$data['success']) {
            set_transient('tiny_king_api_key_validation', 'invalid', DAY_IN_SECONDS);
            return false;
        }
        
        // Store validation status as valid
        set_transient('tiny_king_api_key_validation', 'valid', DAY_IN_SECONDS);
        return true;
    }

    /**
     * Render API settings section
     */
    public function render_api_settings_section() {
        echo '<p>' . __('Configure your TinyKing API settings.', 'tiny-king') . '</p>';
    }

    /**
     * Render API URL field
     */
    public function render_api_url_field() {
        $value = isset($this->settings['api_url']) ? $this->settings['api_url'] : '';
        echo '<input type="url" name="tiny_king_settings[api_url]" value="' . esc_attr($value) . '" class="regular-text" placeholder="https://your-api-url.com/api/wordpress/compress" />';
        echo '<p class="description">' . __('Enter the URL of your TinyKing API endpoint.', 'tiny-king') . '</p>';
    }

    /**
     * Render API key field
     */
    public function render_api_key_field() {
        $value = isset($this->settings['api_key']) ? $this->settings['api_key'] : '';
        $is_valid = false;
        $validation_status = '';
        
        // Check if API key is set and not empty
        if (!empty($value)) {
            // Get validation status from transient
            $validation_status = get_transient('tiny_king_api_key_validation');
            
            if ($validation_status === 'valid') {
                $is_valid = true;
            } elseif ($validation_status === 'invalid') {
                $is_valid = false;
            } else {
                // If no validation status is stored, consider it unvalidated
                $validation_status = 'unvalidated';
            }
        }
        
        echo '<div class="tiny-king-api-key-wrapper">';
        echo '<input type="text" name="tiny_king_settings[api_key]" value="' . esc_attr($value) . '" class="regular-text" id="tiny-king-api-key" />';
        
        // Add validation indicator
        if (!empty($value)) {
            if ($is_valid) {
                echo '<span class="tiny-king-validation-indicator valid dashicons dashicons-yes" title="' . esc_attr__('API key is valid', 'tiny-king') . '"></span>';
            } elseif ($validation_status === 'invalid') {
                echo '<span class="tiny-king-validation-indicator invalid dashicons dashicons-no" title="' . esc_attr__('API key is invalid', 'tiny-king') . '"></span>';
            } else {
                echo '<span class="tiny-king-validation-indicator unknown dashicons dashicons-warning" title="' . esc_attr__('API key has not been validated', 'tiny-king') . '"></span>';
            }
        }
        
        echo '</div>';
        echo '<p class="description">' . __('Enter your TinyKing API key.', 'tiny-king') . '</p>';
    }

    /**
     * Render upload settings section
     */
    public function render_upload_settings_section() {
        echo '<p>' . __('Configure how new image uploads are handled.', 'tiny-king') . '</p>';
    }

    /**
     * Render compression mode field
     */
    public function render_compression_mode_field() {
        $value = isset($this->settings['compression_mode']) ? $this->settings['compression_mode'] : 'background';
        ?>
        <div class="tiny-king-radio-options">
            <label>
                <input type="radio" name="tiny_king_settings[compression_mode]" value="background" <?php checked($value, 'background'); ?> />
                <?php _e('Compress new images in the background (Recommended)', 'tiny-king'); ?>
                <p class="description"><?php _e('This is the fastest method, but can cause issues with some image related plugins.', 'tiny-king'); ?></p>
            </label>
            
            <label>
                <input type="radio" name="tiny_king_settings[compression_mode]" value="upload" <?php checked($value, 'upload'); ?> />
                <?php _e('Compress new images during upload', 'tiny-king'); ?>
                <p class="description"><?php _e('Uploads will take longer, but provides higher compatibility with other plugins.', 'tiny-king'); ?></p>
            </label>
            
            <label>
                <input type="radio" name="tiny_king_settings[compression_mode]" value="manual" <?php checked($value, 'manual'); ?> />
                <?php _e('Do not compress new images automatically', 'tiny-king'); ?>
                <p class="description"><?php _e('Manually select the images you want to compress in the media library.', 'tiny-king'); ?></p>
            </label>
        </div>
        <?php
    }

    /**
     * Render sizes settings section
     */
    public function render_sizes_settings_section() {
        echo '<p>' . __('WordPress generates resized versions of every image. Choose which sizes to compress.', 'tiny-king') . '</p>';
    }

    /**
     * Render image sizes field
     */
    public function render_image_sizes_field() {
        $image_sizes = isset($this->settings['image_sizes']) ? $this->settings['image_sizes'] : [];
        
        // Get all registered image sizes
        $registered_sizes = [];
        
        // Add default WordPress sizes
        $registered_sizes['original'] = __('Original image (overwritten by compressed image)', 'tiny-king');
        $registered_sizes['thumbnail'] = sprintf(__('Thumbnail - %sx%s', 'tiny-king'), get_option('thumbnail_size_w'), get_option('thumbnail_size_h'));
        $registered_sizes['medium'] = sprintf(__('Medium - %sx%s', 'tiny-king'), get_option('medium_size_w'), get_option('medium_size_h'));
        $registered_sizes['medium_large'] = __('Medium large - 768x?', 'tiny-king');
        $registered_sizes['large'] = sprintf(__('Large - %sx%s', 'tiny-king'), get_option('large_size_w'), get_option('large_size_h'));
        $registered_sizes['1536x1536'] = __('1536x1536 - 1536x1536', 'tiny-king');
        $registered_sizes['2048x2048'] = __('2048x2048 - 2048x2048', 'tiny-king');
        
        // Add custom image sizes
        global $_wp_additional_image_sizes;
        if (isset($_wp_additional_image_sizes) && is_array($_wp_additional_image_sizes)) {
            foreach ($_wp_additional_image_sizes as $size_name => $size_data) {
                $width = isset($size_data['width']) ? $size_data['width'] : '?';
                $height = isset($size_data['height']) ? $size_data['height'] : '?';
                $registered_sizes[$size_name] = sprintf('%s - %sx%s', $size_name, $width, $height);
            }
        }
        
        // Output checkboxes for each size
        echo '<div class="tiny-king-image-sizes">';
        foreach ($registered_sizes as $size_name => $size_label) {
            $checked = isset($image_sizes[$size_name]) ? $image_sizes[$size_name] : false;
            echo '<label>';
            echo '<input type="checkbox" name="tiny_king_settings[image_sizes][' . esc_attr($size_name) . ']" value="1" ' . checked($checked, true, false) . ' />';
            echo esc_html($size_label);
            echo '</label><br>';
        }
        echo '</div>';
        
        echo '<p class="description">' . __('Remember each selected size counts as a compression.', 'tiny-king') . '</p>';
        // echo '<p class="description">' . __('With these settings you can compress at least 50 images for free each month.', 'tiny-king') . '</p>';
    }

    /**
     * Render original settings section
     */
    public function render_original_settings_section() {
        // No description needed
    }

    /**
     * Render resize original field
     */
    public function render_resize_original_field() {
        $resize = isset($this->settings['resize_original']) ? $this->settings['resize_original'] : false;
        $max_width = isset($this->settings['max_width']) ? $this->settings['max_width'] : 2048;
        $max_height = isset($this->settings['max_height']) ? $this->settings['max_height'] : 2048;
        
        echo '<label>';
        echo '<input type="checkbox" name="tiny_king_settings[resize_original]" value="1" ' . checked($resize, true, false) . ' />';
        echo '</label>';
        
        echo '<p class="description">' . __('Save space by setting a maximum width and height for all images uploaded.', 'tiny-king') . '</p>';
        echo '<p class="description">' . __('Resizing takes 1 additional compression for each image that is larger.', 'tiny-king') . '</p>';
        
        echo '<div class="tiny-king-resize-dimensions" style="margin-top: 10px;">';
        echo '<label>' . __('Max Width:', 'tiny-king') . ' ';
        echo '<input type="number" name="tiny_king_settings[max_width]" value="' . esc_attr($max_width) . '" min="100" max="5000" step="1" style="width: 100px;" />';
        echo '</label>';
        
        echo '&nbsp;&nbsp;&nbsp;';
        
        echo '<label>' . __('Max Height:', 'tiny-king') . ' ';
        echo '<input type="number" name="tiny_king_settings[max_height]" value="' . esc_attr($max_height) . '" min="100" max="5000" step="1" style="width: 100px;" />';
        echo '</label>';
        echo '</div>';
    }

    /**
     * Render preserve metadata field
     */
    public function render_preserve_metadata_field() {
        $preserve_exif = isset($this->settings['preserve_exif_data']) ? $this->settings['preserve_exif_data'] : false;
        $preserve_copyright = isset($this->settings['preserve_copyright']) ? $this->settings['preserve_copyright'] : false;
        $preserve_gps = isset($this->settings['preserve_gps']) ? $this->settings['preserve_gps'] : false;
        
        echo '<div class="tiny-king-metadata-options">';
        
        echo '<label>';
        echo '<input type="checkbox" name="tiny_king_settings[preserve_exif_data]" value="1" ' . checked($preserve_exif, true, false) . ' />';
        echo __('Preserve creation date and time in the original image', 'tiny-king');
        echo '</label><br>';
        
        echo '<label>';
        echo '<input type="checkbox" name="tiny_king_settings[preserve_copyright]" value="1" ' . checked($preserve_copyright, true, false) . ' />';
        echo __('Preserve copyright information in the original image', 'tiny-king');
        echo '</label><br>';
        
        echo '<label>';
        echo '<input type="checkbox" name="tiny_king_settings[preserve_gps]" value="1" ' . checked($preserve_gps, true, false) . ' />';
        echo __('Preserve GPS location in the original image (JPEG only)', 'tiny-king');
        echo '</label>';
        
        echo '</div>';
    }

    /**
     * Render optimization settings section
     */
    public function render_optimization_settings_section() {
        echo '<p>' . __('Configure your image optimization settings.', 'tiny-king') . '</p>';
    }

    /**
     * Render quality field
     */
    public function render_quality_field() {
        $value = isset($this->settings['quality']) ? $this->settings['quality'] : 65;
        echo '<input type="range" name="tiny_king_settings[quality]" value="' . esc_attr($value) . '" min="10" max="100" step="5" class="tiny-king-range" />';
        echo '<span class="tiny-king-range-value">' . esc_html($value) . '%</span>';
        echo '<p class="description">' . __('Set the quality level for image compression. Lower values (40-70) provide better compression but may affect image quality. Higher values (80-100) preserve quality but result in larger files.', 'tiny-king') . '</p>';
    }

    /**
     * Render format field
     */
    public function render_format_field() {
        $value = isset($this->settings['format']) ? $this->settings['format'] : 'original';
        $formats = [
            'original' => __('Original (keep same format)', 'tiny-king'),
            'jpeg' => __('JPEG', 'tiny-king'),
            'png' => __('PNG', 'tiny-king'),
            'webp' => __('WebP', 'tiny-king'),
            'avif' => __('AVIF', 'tiny-king'),
        ];
        
        echo '<select name="tiny_king_settings[format]">';
        foreach ($formats as $format => $label) {
            echo '<option value="' . esc_attr($format) . '" ' . selected($value, $format, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . __('Choose the output format for compressed images.', 'tiny-king') . '</p>';
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('tiny_king_settings');
                do_settings_sections('tiny_king_settings');
                submit_button(__('Save Settings', 'tiny-king'));
                ?>
            </form>
            
            <div class="tiny-king-test-connection">
                <h2><?php _e('Test Connection', 'tiny-king'); ?></h2>
                <p><?php _e('Test your API connection to ensure everything is working correctly.', 'tiny-king'); ?></p>
                <button type="button" class="button button-secondary" id="tiny-king-test-connection">
                    <?php _e('Test Connection', 'tiny-king'); ?>
                </button>
                <div id="tiny-king-test-result" style="margin-top: 10px; display: none;"></div>
            </div>
        </div>
        <?php
    }

    /**
     * Render bulk optimizer page
     */
    public function render_bulk_optimizer_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Get all image attachments
        $args = [
            'post_type' => 'attachment',
            'post_mime_type' => ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
            'post_status' => 'inherit',
            'posts_per_page' => -1,
        ];
        
        $images = get_posts($args);
        $total_images = count($images);
        
        // Get optimization statistics
        $stats = $this->get_optimization_stats();
        
        // Get settings to show which sizes are selected
        $selected_sizes = isset($this->settings['image_sizes']) ? $this->settings['image_sizes'] : [];
        $selected_size_count = 0;
        foreach ($selected_sizes as $is_selected) {
            if ($is_selected) {
                $selected_size_count++;
            }
        }
        
        ?>
        <div class="wrap tiny-king-bulk-page">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        
            <div class="tiny-king-bulk-container">
                <!-- Top section with statistics -->
                <div class="tiny-king-stats-container">
                    <!-- Available Images -->
                    <div class="tiny-king-stats-box">
                        <h2><?php _e('Available Images', 'tiny-king'); ?></h2>
                        <p><?php _e('Here you can start optimizing your entire library. Press the big button to start improving your website speed instantly!', 'tiny-king'); ?></p>
                    
                        <div class="tiny-king-stats-grid">
                            <div class="tiny-king-stats-item">
                                <h3><?php _e('UPLOADED IMAGES', 'tiny-king'); ?></h3>
                                <div class="tiny-king-stats-number"><?php echo $total_images; ?></div>
                            </div>
                        
                            <div class="tiny-king-stats-item">
                                <h3><?php _e('UNCOMPRESSED IMAGE SIZES', 'tiny-king'); ?></h3>
                                <div class="tiny-king-stats-number">
                                    <?php echo $stats['uncompressed_sizes']; ?>
                                    <span class="tiny-king-info-icon" title="<?php echo sprintf(__('WordPress creates multiple sizes of each image. You have selected %d size(s) to optimize in settings.', 'tiny-king'), $selected_size_count); ?>">?</span>
                                </div>
                            </div>
                        </div>
                    </div>
                
                    <!-- Total Savings -->
                    <div class="tiny-king-stats-box">
                        <h2><?php _e('Total Savings', 'tiny-king'); ?></h2>
                        <p><?php _e('Statistics based on all available JPEG, PNG, and WebP images in your media library.', 'tiny-king'); ?></p>
                    
                        <div class="tiny-king-savings-container">
                            <div class="tiny-king-savings-chart">
                                <div class="tiny-king-chart-circle" data-percent="<?php echo $stats['savings_percent']; ?>">
                                    <div class="tiny-king-chart-inner">
                                        <div class="tiny-king-chart-percent"><?php echo number_format($stats['savings_percent'], 1); ?>%</div>
                                        <div class="tiny-king-chart-text"><?php _e('savings', 'tiny-king'); ?></div>
                                    </div>
                                </div>
                            </div>
                        
                            <div class="tiny-king-savings-details">
                                <div class="tiny-king-savings-count"><?php echo $stats['optimized_sizes']; ?> <?php _e('image sizes optimized', 'tiny-king'); ?></div>
                                <div class="tiny-king-savings-size">
                                    <div class="tiny-king-savings-original"><?php echo size_format($stats['original_size'], 2); ?> <span><?php _e('initial size', 'tiny-king'); ?></span></div>
                                    <div class="tiny-king-savings-compressed"><?php echo size_format($stats['compressed_size'], 2); ?> <span><?php _e('current size', 'tiny-king'); ?></span></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            
                <!-- Reminder message -->
                <div class="tiny-king-reminder">
                    <strong><?php _e('Remember', 'tiny-king'); ?></strong>: <?php _e('For the plugin to do the work, you need to keep this page open. But no worries: when stopped, you can continue where you left off!', 'tiny-king'); ?>
                </div>
            
                <!-- Progress bar and button -->
                <div class="tiny-king-progress-section">
                    <div class="tiny-king-progress-container">
                        <div class="tiny-king-progress">
                            <div class="tiny-king-progress-bar" style="width: <?php echo $stats['progress_percent']; ?>%;"></div>
                        </div>
                        <div class="tiny-king-progress-text">
                            <?php echo $stats['optimized_sizes']; ?> / <?php echo $stats['total_sizes']; ?> (<?php echo round($stats['progress_percent']); ?>%)
                        </div>
                    </div>
                
                    <button type="button" class="button button-primary button-hero" id="tiny-king-bulk-optimize" data-action="<?php echo $stats['is_running'] ? 'cancel' : 'optimize'; ?>">
                        <?php echo $stats['is_running'] ? __('CANCEL', 'tiny-king') : __('OPTIMIZE', 'tiny-king'); ?>
                    </button>
                </div>
            
                <!-- Images table -->
                <div class="tiny-king-images-table-container">
                    <table class="tiny-king-images-table widefat">
                        <thead>
                            <tr>
                                <th><?php _e('File', 'tiny-king'); ?></th>
                                <th><?php _e('Initial Size', 'tiny-king'); ?></th>
                                <th><?php _e('Current Size', 'tiny-king'); ?></th>
                                <th><?php _e('Savings', 'tiny-king'); ?></th>
                                <th><?php _e('Status', 'tiny-king'); ?></th>
                            </tr>
                        </thead>
                        <tbody id="tiny-king-images-list">
                            <!-- Images will be loaded here via AJAX -->
                            <tr>
                                <td colspan="5"><?php _e('Click OPTIMIZE to start the optimization process.', 'tiny-king'); ?></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    <?php
}

/**
 * Get optimization statistics
 */
private function get_optimization_stats() {
    global $wpdb;
    
    // Get settings to determine which image sizes to count
    $settings = $this->settings;
    $selected_sizes = isset($settings['image_sizes']) ? $settings['image_sizes'] : [];
    
    // Get all image attachments
    $args = [
        'post_type' => 'attachment',
        'post_mime_type' => ['image/jpeg', 'image/png', 'image/webp'],
        'post_status' => 'inherit',
        'posts_per_page' => -1,
        'fields' => 'ids',
    ];
    
    $image_ids = get_posts($args);
    $total_images = count($image_ids);
    
    // Initialize stats
    $stats = [
        'original_size' => 0,
        'compressed_size' => 0,
        'optimized_sizes' => 0,
        'uncompressed_sizes' => 0,
        'total_sizes' => 0,
        'savings_percent' => 0,
        'progress_percent' => 0,
        'is_running' => false,
    ];
    
    // Get optimization status from transient
    $optimization_status = get_transient('tiny_king_bulk_optimization_status');
    if ($optimization_status) {
        $stats['is_running'] = $optimization_status['is_running'] ?? false;
    }
    
    // Count total image sizes (original + generated sizes)
    $total_sizes = 0;
    $optimized_sizes = 0;
    $uncompressed_sizes = 0;
    
    foreach ($image_ids as $image_id) {
        $metadata = wp_get_attachment_metadata($image_id);
        $optimization_data = get_post_meta($image_id, '_tiny_king_data', true);
        $is_original_optimized = $optimization_data && isset($optimization_data['optimized']) && $optimization_data['optimized'];
        
        // Count original image if selected in settings
        if (isset($selected_sizes['original']) && $selected_sizes['original']) {
            $total_sizes++;
            
            if ($is_original_optimized) {
                $optimized_sizes++;
                
                // Add to total size calculations
                if (isset($optimization_data['original_size']) && isset($optimization_data['compressed_size'])) {
                    $stats['original_size'] += $optimization_data['original_size'];
                    $stats['compressed_size'] += $optimization_data['compressed_size'];
                }
            } else {
                $uncompressed_sizes++;
                
                // Add to original size using actual file size
                $file_path = get_attached_file($image_id, true);
                if ($file_path && file_exists($file_path)) {
                    $file_size = filesize($file_path);
                    $stats['original_size'] += $file_size;
                    $stats['compressed_size'] += $file_size; // Same as original since not compressed
                }
            }
        }
        
        // Count generated sizes if they exist and are selected in settings
        if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
            foreach ($metadata['sizes'] as $size_name => $size_data) {
                // Only count if this size is selected in settings
                if (isset($selected_sizes[$size_name]) && $selected_sizes[$size_name]) {
                    $total_sizes++;
                    
                    // For simplicity, we consider a size optimized if the original is optimized
                    // In a real implementation, you would track optimization status for each size
                    if ($is_original_optimized) {
                        $optimized_sizes++;
                    } else {
                        $uncompressed_sizes++;
                    }
                }
            }
        }
    }
    
    $stats['optimized_sizes'] = $optimized_sizes;
    $stats['uncompressed_sizes'] = $uncompressed_sizes;
    $stats['total_sizes'] = $total_sizes;
    
    // Calculate savings percentage
    if ($stats['original_size'] > 0) {
        $savings_bytes = $stats['original_size'] - $stats['compressed_size'];
        $stats['savings_percent'] = ($savings_bytes / $stats['original_size']) * 100;
        // Round to 1 decimal place for display
        $stats['savings_percent'] = round($stats['savings_percent'], 1);
    }
    
    // Calculate progress percentage
    if ($stats['total_sizes'] > 0) {
        $stats['progress_percent'] = ($stats['optimized_sizes'] / $stats['total_sizes']) * 100;
    }
    
    return $stats;
}

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'tiny-king') === false && $hook !== 'upload.php') {
            return;
        }
        
        wp_enqueue_style(
            'tiny-king-admin',
            TINY_KING_PLUGIN_URL . 'assets/css/admin.css',
            [],
            TINY_KING_VERSION
        );
        
        wp_enqueue_script(
            'tiny-king-admin',
            TINY_KING_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery'],
            TINY_KING_VERSION,
            true
        );
        
        // Replace the existing wp_localize_script call with this updated version:
        wp_localize_script('tiny-king-admin', 'tinyKingData', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('tiny-king-nonce'),
            'testingConnection' => __('Testing connection...', 'tiny-king'),
            'connectionSuccess' => __('Connection successful!', 'tiny-king'),
            'connectionFailed' => __('Connection failed: ', 'tiny-king'),
            'optimizing' => __('Optimizing...', 'tiny-king'),
            'optimizeSuccess' => __('Image optimized successfully!', 'tiny-king'),
            'optimizeFailed' => __('Optimization failed: ', 'tiny-king'),
            'testConnection' => __('Test Connection', 'tiny-king'),
            'errorMessage' => __('An error occurred.', 'tiny-king'),
            'optimize' => __('Optimize', 'tiny-king'),
            'reOptimize' => __('Re-optimize', 'tiny-king'),
            'bulkOptimizing' => __('Optimizing images...', 'tiny-king'),
            'bulkComplete' => __('Optimization complete!', 'tiny-king'),
            'bulkFailed' => __('Bulk optimization failed.', 'tiny-king'),
            'cancel' => __('CANCEL', 'tiny-king'),
            'originalSize' => __('Original Size:', 'tiny-king'),
            'compressedSize' => __('Compressed Size:', 'tiny-king'),
            'savings' => __('Savings:', 'tiny-king'),
            'optimizedOn' => __('Optimized On:', 'tiny-king'),
            'optimizationInfo' => __('Optimization Info', 'tiny-king'),
            'actualFileSize' => __('Actual File Size:', 'tiny-king'),
            'apiKeyValid' => __('API key is valid', 'tiny-king'),
            'apiKeyInvalid' => __('API key is invalid', 'tiny-king'),
            'apiUrlRequired' => __('API URL is required', 'tiny-king'),
            'connectionError' => __('Connection error', 'tiny-king'),
        ]);
    }

    /**
     * Add compress button to media library
     */
    public function add_compress_button($form_fields, $post) {
        if (!current_user_can('manage_options')) {
            return $form_fields;
        }
        
        // Only add button for image attachments
        if (strpos($post->post_mime_type, 'image') === false) {
            return $form_fields;
        }
        
        // Check if image is already optimized
        $optimization_data = get_post_meta($post->ID, '_tiny_king_data', true);
        $optimized = $optimization_data && isset($optimization_data['optimized']) && $optimization_data['optimized'];
        
        $button_text = $optimized ? __('Re-optimize', 'tiny-king') : __('Optimize', 'tiny-king');
        $button_class = $optimized ? 'tiny-king-reoptimize' : 'tiny-king-optimize';
        $disabled = $optimized ? '' : '';
        
        $form_fields['tiny_king'] = [
            'label' => __('TinyKing', 'tiny-king'),
            'input' => 'html',
            'html' => '<button type="button" class="button button-secondary ' . esc_attr($button_class) . '" data-id="' . esc_attr($post->ID) . '" ' . $disabled . '>' . $button_text . '</button>' .
                      '<div class="tiny-king-status" id="tiny-king-status-' . esc_attr($post->ID) . '"></div>',
            'helps' => __('Optimize this image using TinyKing.', 'tiny-king'),
        ];
        
        return $form_fields;
    }

    /**
     * Add file size information to media row actions
     */
    public function add_file_size_to_media_row($actions, $post) {
        if (strpos($post->post_mime_type, 'image') === false) {
            return $actions;
        }
        
        $file_path = get_attached_file($post->ID, true);
        if ($file_path && file_exists($file_path)) {
            clearstatcache(true, $file_path);
            $file_size = filesize($file_path);
            $actions['file_size'] = '<span class="tiny-king-file-size">Size: ' . size_format($file_size) . '</span>';
        }
        
        return $actions;
    }

    /**
     * Add detailed file information to attachment details
     */
    public function add_file_info_to_attachment($form_fields, $post) {
        if (strpos($post->post_mime_type, 'image') === false) {
            return $form_fields;
        }
        
        $file_path = get_attached_file($post->ID, true);
        if (!$file_path || !file_exists($file_path)) {
            return $form_fields;
        }
        
        clearstatcache(true, $file_path);
        $file_size = filesize($file_path);
        $file_modified = filemtime($file_path);
        
        $form_fields['tiny_king_file_info'] = [
            'label' => __('File Info', 'tiny-king'),
            'input' => 'html',
            'html' => '<div class="tiny-king-file-info">' .
                      '<p><strong>' . __('Actual File Size:', 'tiny-king') . '</strong> ' . size_format($file_size) . '</p>' .
                      '<p><strong>' . __('Last Modified:', 'tiny-king') . '</strong> ' . date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $file_modified) . '</p>' .
                      '<p><strong>' . __('File Path:', 'tiny-king') . '</strong> <code>' . esc_html($file_path) . '</code></p>' .
                      '</div>',
        ];
        
        // Add optimization data if available
        $optimization_data = get_post_meta($post->ID, '_tiny_king_data', true);
        if ($optimization_data && isset($optimization_data['optimized'])) {
            $form_fields['tiny_king_optimization_info'] = [
                'label' => __('Optimization Info', 'tiny-king'),
                'input' => 'html',
                'html' => '<div class="tiny-king-optimization-info">' .
                          '<p><strong>' . __('Original Size:', 'tiny-king') . '</strong> ' . size_format($optimization_data['original_size']) . '</p>' .
                          '<p><strong>' . __('Compressed Size:', 'tiny-king') . '</strong> ' . size_format($optimization_data['compressed_size']) . '</p>' .
                          '<p><strong>' . __('Savings:', 'tiny-king') . '</strong> ' . size_format($optimization_data['savings']) . ' (' . $optimization_data['savings_percent'] . '%)</p>' .
                          '<p><strong>' . __('Optimized On:', 'tiny-king') . '</strong> ' . date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $optimization_data['timestamp']) . '</p>' .
                          '</div>',
            ];
        }
        
        return $form_fields;
    }

    /**
     * Auto-compress uploaded images
     */
    public function auto_compress_uploaded_image($upload) {
        // Only process images
        $file_type = wp_check_filetype($upload['file']);
        $mime_type = $file_type['type'];
        
        if (!$mime_type || strpos($mime_type, 'image') === false) {
            return $upload;
        }
        
        // Get attachment ID
        $attachment = $this->get_attachment_by_file($upload['file']);
        if (!$attachment) {
            return $upload;
        }
        
        // Compress the image
        $result = $this->compress_attachment($attachment->ID);
        
        // Return the original upload data
        return $upload;
    }

    /**
     * Get attachment by file path
     */
    private function get_attachment_by_file($file) {
        $file = str_replace(wp_upload_dir()['basedir'] . '/', '', $file);
        
        $args = [
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => 1,
            'meta_query' => [
                [
                    'key' => '_wp_attached_file',
                    'value' => $file,
                    'compare' => '=',
                ],
            ],
        ];
        
        $attachments = get_posts($args);
        
        if (!empty($attachments)) {
            return $attachments[0];
        }
        
        return null;
    }

    /**
     * Compress an attachment
     */
    public function compress_attachment($attachment_id) {
        // Get attachment file path
        $file_path = get_attached_file($attachment_id, true);
        if (!$file_path || !file_exists($file_path)) {
            return new WP_Error('file_not_found', __('Attachment file not found.', 'tiny-king'));
        }
        
        // Check API settings
        if (empty($this->api_url) || empty($this->api_key)) {
            return new WP_Error('api_not_configured', __('API URL and API key must be configured.', 'tiny-king'));
        }
        
        // Check if image is already optimized
        $optimization_data = get_post_meta($attachment_id, '_tiny_king_data', true);
        if ($optimization_data && isset($optimization_data['optimized']) && $optimization_data['optimized']) {
            // Image is already optimized, return the existing data
            return $optimization_data;
        }
        
        // Get file info
        $file_name = basename($file_path);
        $original_size = filesize($file_path);
        $mime_type = mime_content_type($file_path);
        
        // Resize original if enabled
        if ($this->settings['resize_original']) {
            $this->resize_original_image($file_path, $this->settings['max_width'], $this->settings['max_height']);
        }
        
        // Use cURL for direct file upload
        $curl = curl_init();
        
        // Create a CURLFile object
        $cfile = curl_file_create($file_path, $mime_type, $file_name);
        
        // Set up the POST fields
        $post_fields = [
            'image' => $cfile,
            'format' => isset($this->settings['format']) ? $this->settings['format'] : 'original',
            'quality' => isset($this->settings['quality']) ? $this->settings['quality'] : 65,
            'stripMetadata' => !($this->settings['preserve_exif_data'] || $this->settings['preserve_copyright'] || $this->settings['preserve_gps']),
            'forceCompression' => 'true'
        ];
        
        // Set cURL options
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->api_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $post_fields,
            CURLOPT_HTTPHEADER => [
                'x-api-key: ' . $this->api_key,
            ],
            CURLOPT_TIMEOUT => 60,
            CURLOPT_SSL_VERIFYPEER => apply_filters('tiny_king_sslverify', true),
        ]);
        
        // Execute the request
        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($curl);
        
        curl_close($curl);
        
        // Check for cURL errors
        if ($response === false) {
            return new WP_Error('curl_error', $curl_error);
        }
        
        // Check HTTP status code
        if ($http_code !== 200) {
            return new WP_Error('http_error', sprintf(__('HTTP error: %d', 'tiny-king'), $http_code));
        }
        
        // Parse the JSON response
        $data = json_decode($response, true);
        
        if (!$data || !isset($data['success']) || !$data['success']) {
            $error = isset($data['error']) ? $data['error'] : __('Unknown error', 'tiny-king');
            if (isset($data['details'])) {
                $error .= ': ' . $data['details'];
            }
            return new WP_Error('api_response_error', $error);
        }
        
        // Get the compressed image data from base64
        if (!isset($data['base64Data']) || empty($data['base64Data'])) {
            return new WP_Error('compressed_data_error', __('Could not retrieve compressed image data.', 'tiny-king'));
        }
        
        $base64_data = $data['base64Data'];
        // Remove data URL prefix if present
        if (strpos($base64_data, 'data:') === 0) {
            $base64_data = substr($base64_data, strpos($base64_data, ',') + 1);
        }
        $compressed_data = base64_decode($base64_data);
        
        // Determine output path
        $output_path = $file_path;
        
        // If format changed, update the file extension
        if (isset($this->settings['format']) && $this->settings['format'] !== 'original' && isset($data['format'])) {
            $new_extension = $data['format'];
            $output_path = preg_replace('/\.[^.]+$/', '.' . $new_extension, $file_path);
        }
        
        // Create a backup of the original file
        $backup_path = $file_path . '.bak';
        if (!file_exists($backup_path)) {
            copy($file_path, $backup_path);
        }
        
        // Save the compressed image
        $saved = file_put_contents($output_path, $compressed_data);
        if (!$saved) {
            return new WP_Error('file_write_error', __('Could not save compressed image.', 'tiny-king'));
        }
        
        // Verify the file was actually written and get its size
        clearstatcache(true, $output_path);
        $compressed_size = filesize($output_path);
        
        // If the compression didn't reduce the file size, restore the original
        if ($compressed_size >= $original_size) {
            // Restore the original file
            if (file_exists($backup_path)) {
                copy($backup_path, $file_path);
            }
            
            return [
                'optimized' => false,
                'original_size' => $original_size,
                'compressed_size' => $original_size,
                'savings' => 0,
                'savings_percent' => 0,
                'timestamp' => time(),
            ];
        }
        
        // If the file path changed (due to format conversion), update the attachment metadata
        if ($output_path !== $file_path) {
            update_attached_file($attachment_id, $output_path);
        }
        
        // Force WordPress to regenerate attachment metadata
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $metadata = wp_generate_attachment_metadata($attachment_id, $output_path);
        wp_update_attachment_metadata($attachment_id, $metadata);
        
        // Calculate savings
        $savings = $original_size - $compressed_size;
        $savings_percent = round(($savings / $original_size) * 100, 1); // Round to 1 decimal place
        
        // Store optimization data in attachment meta
        $optimization_data = [
            'optimized' => true,
            'original_size' => $original_size,
            'compressed_size' => $compressed_size,
            'savings' => $savings,
            'savings_percent' => $savings_percent,
            'timestamp' => time(),
        ];
        
        update_post_meta($attachment_id, '_tiny_king_data', $optimization_data);
        
        // Clear any WordPress caches that might be storing the old file size
        clean_attachment_cache($attachment_id);
        
        return $optimization_data;
    }

    /**
     * Resize original image if it exceeds max dimensions
     */
    private function resize_original_image($file_path, $max_width, $max_height) {
        if (!function_exists('wp_get_image_editor')) {
            return false;
        }
        
        $editor = wp_get_image_editor($file_path);
        if (is_wp_error($editor)) {
            return false;
        }
        
        $size = $editor->get_size();
        $width = $size['width'];
        $height = $size['height'];
        
        // Only resize if the image exceeds max dimensions
        if ($width <= $max_width && $height <= $max_height) {
            return false;
        }
        
        $resized = $editor->resize($max_width, $max_height, false);
        if (is_wp_error($resized)) {
            return false;
        }
        
        $saved = $editor->save($file_path);
        return !is_wp_error($saved);
    }
}

// Initialize the plugin
$tiny_king = new TinyKing();
